import React, { createContext, useContext, useState, useCallback, ReactNode } from 'react';
import { BuildingConfig, Module, ModuleType, GridPosition, Floor, ModuleConfig } from './types';

interface InputConfigContextType {
  buildingConfig: BuildingConfig;
  isDrawerOpen: boolean;
  selectedModule: Module | null;
  openDrawer: () => void;
  closeDrawer: () => void;
  selectModule: (module: Module | null) => void;
  addModule: (floorId: string, position: GridPosition, type: ModuleType) => void;
  removeModule: (floorId: string, position: GridPosition) => void;
  updateModule: (moduleId: string, config: ModuleConfig) => void;
  toggleModuleCollapse: (moduleId: string) => void;
  updateBuildingConfig: (config: BuildingConfig) => void;
}

const InputConfigContext = createContext<InputConfigContextType | undefined>(undefined);

const initialBuildingConfig: BuildingConfig = {
  floors: [
    {
      id: 'floor-1',
      name: 'Floor 1',
      modules: Array(4).fill(null).map(() => Array(4).fill(null)),
    },
    {
      id: 'floor-2',
      name: 'Floor 2',
      modules: Array(4).fill(null).map(() => Array(4).fill(null)),
    },
    {
      id: 'floor-3',
      name: 'Floor 3',
      modules: Array(4).fill(null).map(() => Array(4).fill(null)),
    },
  ],
};

export function InputConfigProvider({ children }: { children: ReactNode }) {
  const [buildingConfig, setBuildingConfig] = useState<BuildingConfig>(initialBuildingConfig);
  const [isDrawerOpen, setIsDrawerOpen] = useState(false);
  const [selectedModule, setSelectedModule] = useState<Module | null>(null);

  const openDrawer = useCallback(() => {
    setIsDrawerOpen(true);
  }, []);

  const closeDrawer = useCallback(() => {
    setIsDrawerOpen(false);
    setSelectedModule(null);
  }, []);

  const selectModule = useCallback((module: Module | null) => {
    setSelectedModule(module);
  }, []);

  const addModule = useCallback((floorId: string, position: GridPosition, type: ModuleType) => {
    setBuildingConfig((prev) => {
      const newConfig = { ...prev };
      const floor = newConfig.floors.find((f) => f.id === floorId);
      if (!floor) return prev;

      const moduleId = `module-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;
      const newModule: Module = {
        id: moduleId,
        type,
        config: {},
        position,
        floorId,
        collapsed: false,
      };

      const newModules = floor.modules.map((row, rowIdx) =>
        row.map((cell, colIdx) => {
          if (rowIdx === position.row && colIdx === position.col) {
            return newModule;
          }
          return cell;
        })
      );

      floor.modules = newModules;
      return newConfig;
    });
  }, []);

  const removeModule = useCallback((floorId: string, position: GridPosition) => {
    setBuildingConfig((prev) => {
      const newConfig = { ...prev };
      const floor = newConfig.floors.find((f) => f.id === floorId);
      if (!floor) return prev;

      const newModules = floor.modules.map((row, rowIdx) =>
        row.map((cell, colIdx) => {
          if (rowIdx === position.row && colIdx === position.col) {
            return null;
          }
          return cell;
        })
      );

      floor.modules = newModules;
      return newConfig;
    });
    setSelectedModule(null);
  }, []);

  const updateModule = useCallback((moduleId: string, config: ModuleConfig) => {
    setBuildingConfig((prev) => {
      const newConfig = { ...prev };
      newConfig.floors.forEach((floor) => {
        floor.modules.forEach((row) => {
          row.forEach((module) => {
            if (module && module.id === moduleId) {
              module.config = { ...module.config, ...config };
            }
          });
        });
      });
      return newConfig;
    });

    if (selectedModule && selectedModule.id === moduleId) {
      setSelectedModule({ ...selectedModule, config: { ...selectedModule.config, ...config } });
    }
  }, [selectedModule]);

  const toggleModuleCollapse = useCallback((moduleId: string) => {
    setBuildingConfig((prev) => {
      const newConfig = { ...prev };
      newConfig.floors.forEach((floor) => {
        floor.modules.forEach((row) => {
          row.forEach((module) => {
            if (module && module.id === moduleId) {
              module.collapsed = !module.collapsed;
            }
          });
        });
      });
      return newConfig;
    });
  }, []);

  const updateBuildingConfig = useCallback((config: BuildingConfig) => {
    setBuildingConfig(config);
  }, []);

  return (
    <InputConfigContext.Provider
      value={{
        buildingConfig,
        isDrawerOpen,
        selectedModule,
        openDrawer,
        closeDrawer,
        selectModule,
        addModule,
        removeModule,
        updateModule,
        toggleModuleCollapse,
        updateBuildingConfig,
      }}
    >
      {children}
    </InputConfigContext.Provider>
  );
}

export function useInputConfig() {
  const context = useContext(InputConfigContext);
  if (context === undefined) {
    throw new Error('useInputConfig must be used within an InputConfigProvider');
  }
  return context;
}

