import { useState } from 'react';
import { Thermometer, Wind, Droplets, Users, GripVertical } from 'lucide-react';
import { SensorConfig, SensorType } from '../../types';
import { DndContext, closestCenter, KeyboardSensor, PointerSensor, useSensor, useSensors, DragEndEvent } from '@dnd-kit/core';
import { arrayMove, SortableContext, sortableKeyboardCoordinates, useSortable, verticalListSortingStrategy } from '@dnd-kit/sortable';
import { CSS } from '@dnd-kit/utilities';

const availableSensors: SensorType[] = [
  { id: 'temp', name: 'Temperature', icon: 'thermometer', enabled: false, priority: 0 },
  { id: 'co2', name: 'CO₂', icon: 'wind', enabled: false, priority: 0 },
  { id: 'humidity', name: 'Humidity', icon: 'droplets', enabled: false, priority: 0 },
  { id: 'occupancy', name: 'Occupancy', icon: 'users', enabled: false, priority: 0 },
];

const iconMap: Record<string, any> = {
  thermometer: Thermometer,
  wind: Wind,
  droplets: Droplets,
  users: Users,
};

interface SortableSensorItemProps {
  sensor: SensorType;
  onToggle: (id: string) => void;
}

function SortableSensorItem({ sensor, onToggle }: SortableSensorItemProps) {
  const { attributes, listeners, setNodeRef, transform, transition, isDragging } = useSortable({
    id: sensor.id,
  });

  const style = {
    transform: CSS.Transform.toString(transform),
    transition,
    opacity: isDragging ? 0.5 : 1,
  };

  const Icon = iconMap[sensor.icon] || Thermometer;

  return (
    <div
      ref={setNodeRef}
      style={style}
      className="flex items-center gap-3 p-3 bg-white border border-gray-200 rounded-lg hover:shadow-md transition"
    >
      <div {...attributes} {...listeners} className="cursor-grab active:cursor-grabbing">
        <GripVertical className="w-5 h-5 text-gray-400" />
      </div>
      <input
        type="checkbox"
        checked={sensor.enabled}
        onChange={() => onToggle(sensor.id)}
        className="w-4 h-4 text-primary rounded focus:ring-primary"
      />
      <Icon className="w-5 h-5 text-gray-600" />
      <span className="flex-1 text-sm font-medium text-gray-700">{sensor.name}</span>
      <span className="text-xs text-gray-500">Priority: {sensor.priority + 1}</span>
    </div>
  );
}

interface SensorTypeSelectorProps {
  config: SensorConfig;
  onUpdate: (config: SensorConfig) => void;
}

export function SensorTypeSelector({ config, onUpdate }: SensorTypeSelectorProps) {
  const [sensors, setSensors] = useState<SensorType[]>(
    config.selectedSensors?.length > 0
      ? config.selectedSensors
      : availableSensors
  );

  const dndSensors = useSensors(
    useSensor(PointerSensor),
    useSensor(KeyboardSensor, {
      coordinateGetter: sortableKeyboardCoordinates,
    })
  );

  const handleToggle = (id: string) => {
    const updated = sensors.map((s) =>
      s.id === id ? { ...s, enabled: !s.enabled } : s
    );
    setSensors(updated);
    onUpdate({ ...config, selectedSensors: updated });
  };

  const handleDragEnd = (event: DragEndEvent) => {
    const { active, over } = event;
    if (over && active.id !== over.id) {
      const oldIndex = sensors.findIndex((s) => s.id === active.id);
      const newIndex = sensors.findIndex((s) => s.id === over.id);
      const updated = arrayMove(sensors, oldIndex, newIndex).map((s, idx) => ({
        ...s,
        priority: idx,
      }));
      setSensors(updated);
      onUpdate({ ...config, selectedSensors: updated });
    }
  };

  return (
    <div className="space-y-4">
      <p className="text-sm text-gray-600 mb-4">
        Select sensor types and reorder by priority (drag to reorder)
      </p>
      <DndContext
        sensors={dndSensors}
        collisionDetection={closestCenter}
        onDragEnd={handleDragEnd}
      >
        <SortableContext items={sensors.map((s) => s.id)} strategy={verticalListSortingStrategy}>
          <div className="space-y-2">
            {sensors.map((sensor) => (
              <SortableSensorItem key={sensor.id} sensor={sensor} onToggle={handleToggle} />
            ))}
          </div>
        </SortableContext>
      </DndContext>
    </div>
  );
}

