import { ModuleCard } from './ModuleCard';
import { Floor, Module, GridPosition } from '../../types';
import { useGridCellDroppable } from '../dnd/useDragAndDrop';
import { useInputConfig } from '../store';

interface FloorGridProps {
  floor: Floor;
}

interface GridCellProps {
  floorId: string;
  position: GridPosition;
  module: Module | null;
  cellId: string;
}

function GridCell({ floorId, position, module, cellId }: GridCellProps) {
  const { selectModule } = useInputConfig();
  const { ref, isOver } = useGridCellDroppable(cellId, floorId, position);

  return (
    <div
      ref={ref}
      className={`min-h-[80px] p-2 border-2 rounded-lg transition-all ${
        isOver
          ? 'border-primary bg-primary/10'
          : 'border-gray-200 bg-gray-50 hover:border-gray-300'
      }`}
    >
      {module ? (
        <ModuleCard
          module={module}
          onSelect={() => selectModule(module)}
        />
      ) : (
        <div className="w-full h-full flex items-center justify-center text-gray-400 text-xs">
          Drop here
        </div>
      )}
    </div>
  );
}

export function FloorGrid({ floor }: FloorGridProps) {
  return (
    <div className="space-y-4">
      <h3 className="text-sm font-semibold text-gray-700">{floor.name}</h3>
      <div className="grid grid-cols-4 gap-2">
        {floor.modules.map((row, rowIdx) =>
          row.map((module, colIdx) => {
            const cellId = `${floor.id}-${rowIdx}-${colIdx}`;
            return (
              <GridCell
                key={cellId}
                floorId={floor.id}
                position={{ row: rowIdx, col: colIdx }}
                module={module}
                cellId={cellId}
              />
            );
          })
        )}
      </div>
    </div>
  );
}

