import React, { useEffect, useRef } from 'react';
import Chart from 'chart.js/auto';

interface ZonePanelProps {
    selectedZone: any;
    zoneData: any;
    onApplyRecommendation: (recId: string) => void;
}

const ZonePanel: React.FC<ZonePanelProps> = ({ selectedZone, zoneData, onApplyRecommendation }) => {
    const chartRef = useRef<HTMLCanvasElement>(null);
    const chartInstance = useRef<Chart | null>(null);

    useEffect(() => {
        if (chartRef.current && selectedZone) {
            if (chartInstance.current) {
                chartInstance.current.destroy();
            }

            const ctx = chartRef.current.getContext('2d');
            if (ctx) {
                chartInstance.current = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: ['00:00', '04:00', '08:00', '12:00', '16:00', '20:00'],
                        datasets: [{
                            label: 'CO2 (ppm)',
                            data: [420, 410, 450, 800, 750, 600], // Sample data
                            borderColor: '#10B981',
                            tension: 0.4,
                            borderWidth: 2,
                            pointRadius: 0,
                            fill: true,
                            backgroundColor: 'rgba(16, 185, 129, 0.1)'
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: { legend: { display: false } },
                        scales: { x: { display: false }, y: { display: false } }
                    }
                });
            }
        }
        return () => {
            if (chartInstance.current) {
                chartInstance.current.destroy();
            }
        };
    }, [selectedZone]);

    if (!selectedZone) {
        return (
            <div className="w-full h-full flex items-center justify-center text-gray-400 bg-white border-l border-gray-200">
                <p>Select a zone to view details</p>
            </div>
        );
    }

    return (
        <div className="w-80 bg-white border-l border-gray-200 h-full flex flex-col overflow-y-auto shadow-xl z-20">
            <div className="p-6 border-b border-gray-100">
                <h2 className="text-xl font-bold text-gray-800">{selectedZone.label}</h2>
                <span className="text-xs text-gray-500 uppercase tracking-wider">Live Metrics</span>
            </div>

            <div className="p-6 space-y-6">
                {/* Metrics Grid */}
                <div className="grid grid-cols-2 gap-4">
                    <div className="bg-gray-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-500 block">CO2</span>
                        <span className="text-lg font-bold text-gray-900">830 <span className="text-xs font-normal text-gray-500">ppm</span></span>
                    </div>
                    <div className="bg-gray-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-500 block">Temp</span>
                        <span className="text-lg font-bold text-gray-900">22.4 <span className="text-xs font-normal text-gray-500">°C</span></span>
                    </div>
                    <div className="bg-gray-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-500 block">VOC</span>
                        <span className="text-lg font-bold text-gray-900">120 <span className="text-xs font-normal text-gray-500">ppb</span></span>
                    </div>
                    <div className="bg-gray-50 p-3 rounded-lg">
                        <span className="text-xs text-gray-500 block">Humidity</span>
                        <span className="text-lg font-bold text-gray-900">45 <span className="text-xs font-normal text-gray-500">%</span></span>
                    </div>
                </div>

                {/* Sparkline */}
                <div>
                    <h3 className="text-sm font-semibold text-gray-700 mb-2">24h Trend (CO2)</h3>
                    <div className="h-24 w-full">
                        <canvas ref={chartRef}></canvas>
                    </div>
                </div>

                {/* Emission Sources */}
                <div>
                    <h3 className="text-sm font-semibold text-gray-700 mb-3">Top Emission Sources</h3>
                    <div className="space-y-3">
                        <div className="flex items-center justify-between text-sm">
                            <span className="text-gray-600">Chiller Plant</span>
                            <span className="font-medium text-red-600">36%</span>
                        </div>
                        <div className="w-full bg-gray-100 rounded-full h-2">
                            <div className="bg-red-500 h-2 rounded-full" style={{ width: '36%' }}></div>
                        </div>
                    </div>
                </div>

                {/* AI Recommendations */}
                <div>
                    <h3 className="text-sm font-semibold text-gray-700 mb-3 flex items-center gap-2">
                        <i className="fa-solid fa-robot text-primary"></i> AI Recommendations
                    </h3>
                    <div className="space-y-3">
                        <div className="border border-green-100 bg-green-50 p-3 rounded-lg">
                            <div className="flex justify-between items-start mb-1">
                                <span className="text-xs font-bold text-green-700 bg-green-200 px-1.5 py-0.5 rounded">HIGH IMPACT</span>
                                <span className="text-xs text-green-600">78% Conf.</span>
                            </div>
                            <p className="text-sm text-gray-800 font-medium mb-2">Reduce chiller delta-T by 1.2°C</p>
                            <button
                                onClick={() => onApplyRecommendation('rec-1')}
                                className="w-full py-1.5 bg-white border border-green-200 text-green-700 text-xs font-semibold rounded hover:bg-green-100 transition"
                            >
                                Apply Optimization
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default ZonePanel;
